/*
 * Copyright (c) 2020 - 2021 Legacy Fabric
 * Copyright (c) 2016 - 2021 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.content.registries;

import java.util.HashMap;
import java.util.Map;

import com.google.common.collect.BiMap;
import com.google.common.collect.Maps;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.fabricmc.api.ModInitializer;
import net.fabricmc.fabric.api.event.registry.v1.FabricRegistryEntryAddedEvents;
import net.fabricmc.fabric.mixin.content.registries.MutableRegistryAccessor;
import net.fabricmc.fabric.mixin.content.registries.BlockAccessor;
import net.fabricmc.fabric.mixin.content.registries.BlockEntityAccessor;
import net.fabricmc.fabric.mixin.content.registries.EntityTypeAccessor;
import net.fabricmc.fabric.mixin.content.registries.SimpleRegistryAccessor;
import net.minecraft.class_1389;
import net.minecraft.class_160;
import net.minecraft.class_1605;
import net.minecraft.class_1711;
import net.minecraft.class_1745;
import net.minecraft.class_2054;
import net.minecraft.class_348;
import net.minecraft.class_376;

public final class ContentRegistryImpl implements ModInitializer {
	private static final Logger LOGGER = LogManager.getLogger();
	private static final Map<String, Class<? extends class_1745>> MODDED_ENTITIES = new HashMap<>();
	private static final Map<class_1605, class_2054> UNSORTED_ITEMS = new HashMap<>();
	private static final Map<class_1605, class_160> UNSORTED_BLOCKS = new HashMap<>();
	private static final Map<String, Class<? extends class_1745>> UNSORTED_ENTITIES = new HashMap<>();
	private static final class_1389<class_1711<class_1605, class_160>> VANILLA_BLOCKS = new class_1389<>();
	private static final class_1389<class_1711<class_1605, class_2054>> VANILLA_ITEMS = new class_1389<>();
	private static final class_1389<class_376> VANILLA_BLOCK_STATES = new class_1389<>();
	private static final class_1389<class_1711<String, Class<? extends class_1745>>> VANILLA_ENTITIES = new class_1389<>();
	private static int unorderedNextBlockId = 198;
	private static int unorderedNextItemId = 4096;
	private static int unorderedNextEntityId = 201;
	public static boolean blockIdsSetup = false;
	public static boolean itemIdsSetup = false;
	public static boolean entityIdsSetup = false;

	static {
		preserveVanillaEntries();
	}

	private static void preserveVanillaEntries() {
		for (class_160 block : class_160.field_593) {
			VANILLA_BLOCKS.method_4945(new class_1711<>(class_160.field_593.method_4953(block), block), class_160.field_593.method_4952(block));
		}

		for (class_2054 item : class_2054.field_8633) {
			VANILLA_ITEMS.method_4945(new class_1711<>(class_2054.field_8633.method_4953(item), item), class_2054.field_8633.method_4952(item));
		}

		for (class_376 state : class_160.field_594) {
			VANILLA_BLOCK_STATES.method_4945(state, class_160.field_594.method_4946(state));
		}

		for (Map.Entry<String, Class<? extends class_1745>> entry : EntityTypeAccessor.getNameClassMap().entrySet()) {
			VANILLA_ENTITIES.method_4945(new class_1711<>(entry.getKey(), entry.getValue()), EntityTypeAccessor.getNameIdMap().get(entry.getKey()));
		}
	}

	public static <T extends class_160> T registerBlock(class_1605 id, T block) {
		FabricRegistryEntryAddedEvents.BLOCK.invoker().blockAdded(id, block);
		UNSORTED_BLOCKS.put(id, block);
		class_160.field_593.method_4951(unorderedNextBlockId, id, block);

		for (class_376 state : block.method_630().method_1228()) {
			class_160.field_594.method_4945(state, unorderedNextBlockId << 4 | block.method_706(state));
		}

		unorderedNextBlockId++;
		return block;
	}

	public static <T extends class_2054> T registerItem(class_1605 id, T item) {
		FabricRegistryEntryAddedEvents.ITEM.invoker().itemAdded(id, item);
		UNSORTED_ITEMS.put(id, item);
		class_2054.field_8633.method_4951(unorderedNextItemId, id, item);
		unorderedNextItemId++;
		return item;
	}

	public static void registerEntity(Class<? extends class_1745> clazz, String name) {
		if (EntityTypeAccessor.getNameClassMap().containsKey(name)) {
			throw new UnsupportedOperationException(String.format("Duplicate entity name %s found of class %s. If you're trying to overwrite an entity, you can't yet.", name, clazz.getName()));
		} else if (EntityTypeAccessor.getNameClassMap().containsValue(clazz)) {
			throw new UnsupportedOperationException(String.format("Duplicate entity class %s found of name %s. If you're trying to overwrite an entity, you can't yet.", clazz.getName(), name));
		}

		FabricRegistryEntryAddedEvents.ENTITY.invoker().entityAdded(clazz, name);
		UNSORTED_ENTITIES.put(name, clazz);
		EntityTypeAccessor.invokeRegisterEntity(clazz, name, unorderedNextEntityId);
		MODDED_ENTITIES.put(name, clazz);
		unorderedNextEntityId++;
	}

	public static void registerBlockEntity(Class<? extends class_348> clazz, String name) {
		if (BlockEntityAccessor.getStringClassMap().containsKey(name)) {
			throw new UnsupportedOperationException(String.format("Duplicate block entity name %s found of class %s. If you're trying to overwrite a block entity, you can't yet.", name, clazz.getName()));
		} else if (BlockEntityAccessor.getStringClassMap().containsValue(clazz)) {
			throw new UnsupportedOperationException(String.format("Duplicate block entity class %s found of name %s. If you're trying to overwrite a block entity, you can't yet.", clazz.getName(), name));
		}

		BlockEntityAccessor.invokeRegisterBlockEntity(clazz, name);
		FabricRegistryEntryAddedEvents.BLOCK_ENTITY.invoker().blockEntityAdded(clazz, name);
	}

	public static void fillBlocksMapWithUnknownEntries(BiMap<Integer, class_1605> idMap) {
		idMap.values().removeIf(identifier -> !UNSORTED_BLOCKS.containsKey(identifier));

		for (Map.Entry<class_1605, class_160> entry : UNSORTED_BLOCKS.entrySet()) {
			if (!idMap.containsValue(entry.getKey())) {
				int id = nextAvailableBlockId(idMap);
				idMap.put(id, entry.getKey());
			}
		}
	}

	public static void fillItemsMapWithUnknownEntries(BiMap<Integer, class_1605> idMap) {
		idMap.values().removeIf(identifier -> !UNSORTED_BLOCKS.containsKey(identifier));

		for (Map.Entry<class_1605, class_2054> entry : UNSORTED_ITEMS.entrySet()) {
			if (!idMap.containsValue(entry.getKey())) {
				int id = nextAvailableItemId(idMap);
				idMap.put(id, entry.getKey());
			}
		}
	}

	public static void fillEntitiesMapWithUnknownEntries(BiMap<Integer, String> idMap) {
		idMap.values().removeIf(string -> !UNSORTED_ENTITIES.containsKey(string));

		for (Map.Entry<String, Class<? extends class_1745>> entry : UNSORTED_ENTITIES.entrySet()) {
			if (!idMap.containsValue(entry.getKey())) {
				int id = nextAvailableEntityId(idMap);
				idMap.put(id, entry.getKey());
			}
		}
	}

	private static int nextAvailableBlockId(Map<Integer, class_1605> idMap) {
		int i = 198;

		while (true) {
			if (!idMap.containsKey(i)) {
				return i;
			}

			i++;
		}
	}

	private static int nextAvailableItemId(Map<Integer, class_1605> idMap) {
		int i = 432;

		while (true) {
			if ((i < 2256 || i > 2267) && !idMap.containsKey(i)) {
				return i;
			}

			i++;
		}
	}

	private static int nextAvailableEntityId(Map<Integer, String> idMap) {
		int i = 201;

		while (true) {
			if (!idMap.containsKey(i)) {
				return i;
			}

			i++;
		}
	}

	public static void reorderBlockEntries(BiMap<Integer, class_1605> idMap) {
		((MutableRegistryAccessor) class_160.field_593).getMap().clear();
		((SimpleRegistryAccessor) class_160.field_593).setIds(new class_1389<class_160>());
		((SimpleRegistryAccessor) class_160.field_593).getObjects().clear();

		for (class_1711<class_1605, class_160> pair : VANILLA_BLOCKS) {
			class_160.field_593.method_4951(VANILLA_BLOCKS.method_4946(pair), pair.method_6686(), pair.method_6687());
		}

		class_1389<class_376> states = new class_1389<>();

		for (class_376 state : VANILLA_BLOCK_STATES) {
			states.method_4945(state, VANILLA_BLOCK_STATES.method_4946(state));
		}

		for (Map.Entry<Integer, class_1605> entry : idMap.entrySet()) {
			class_160 block = UNSORTED_BLOCKS.get(entry.getValue());
			class_160.field_593.method_4951(entry.getKey(), entry.getValue(), block);

			for (class_376 state : block.method_630().method_1228()) {
				states.method_4945(state, unorderedNextBlockId << 4 | block.method_706(state));
			}
		}

		BlockAccessor.setBlockStates(states);
	}

	public static void reorderItemEntries(BiMap<Integer, class_1605> idMap) {
		((MutableRegistryAccessor) class_2054.field_8633).getMap().clear();
		((SimpleRegistryAccessor) class_2054.field_8633).setIds(new class_1389<class_2054>());
		((SimpleRegistryAccessor) class_2054.field_8633).getObjects().clear();

		for (class_1711<class_1605, class_2054> pair : VANILLA_ITEMS) {
			class_2054.field_8633.method_4951(VANILLA_ITEMS.method_4946(pair), pair.method_6686(), pair.method_6687());
		}

		for (Map.Entry<Integer, class_1605> entry : idMap.entrySet()) {
			class_2054.field_8633.method_4951(entry.getKey(), entry.getValue(), UNSORTED_ITEMS.get(entry.getValue()));
		}
	}

	public static void reorderEntityEntries(BiMap<Integer, String> idMap) {
		EntityTypeAccessor.getNameIdMap().clear();
		EntityTypeAccessor.getClassIdMap().clear();
		EntityTypeAccessor.getIdClassMap().clear();
		EntityTypeAccessor.setNameIdMap(Maps.newHashMap());
		EntityTypeAccessor.setClassIdMap(Maps.newHashMap());
		EntityTypeAccessor.setIdClassMap(Maps.newHashMap());

		for (class_1711<String, Class<? extends class_1745>> pair : VANILLA_ENTITIES) {
			EntityTypeAccessor.getNameIdMap().put(pair.method_6686(), VANILLA_ENTITIES.method_4946(pair));
			EntityTypeAccessor.getClassIdMap().put(pair.method_6687(), VANILLA_ENTITIES.method_4946(pair));
			EntityTypeAccessor.getIdClassMap().put(VANILLA_ENTITIES.method_4946(pair), pair.method_6687());
		}

		for (Map.Entry<Integer, String> entry : idMap.entrySet()) {
			EntityTypeAccessor.getNameIdMap().put(entry.getValue(), entry.getKey());
			EntityTypeAccessor.getClassIdMap().put(MODDED_ENTITIES.get(entry.getValue()), entry.getKey());
			EntityTypeAccessor.getIdClassMap().put(entry.getKey(), MODDED_ENTITIES.get(entry.getValue()));
		}
	}

	@Override
	public void onInitialize() {
	}
}
