/*
 * Copyright (c) 2020 - 2021 Legacy Fabric
 * Copyright (c) 2016 - 2021 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.event.lifecycle.v1;

import javax.annotation.Nullable;
import net.fabricmc.fabric.api.event.Event;
import net.fabricmc.fabric.api.event.EventFactory;
import net.minecraft.class_1635;
import net.minecraft.class_1705;
import net.minecraft.class_1733;
import net.minecraft.class_1745;
import net.minecraft.class_1752;
import net.minecraft.class_1890;
import net.minecraft.class_99;

public final class ServerEntityEvents {
	private ServerEntityEvents() {
	}

	/**
	 * Called when an Entity is loaded into a ServerWorld.
	 *
	 * <p>When this event is called, the entity is already in the world.
	 *
	 * <p>Note there is no corresponding unload event because entity unloads cannot be reliably tracked.
	 */
	public static final Event<Load> ENTITY_LOAD = EventFactory.createArrayBacked(Load.class, callbacks -> (entity, world) -> {
		if (EventFactory.isProfilingEnabled()) {
			final class_1705 profiler = world.field_250;
			profiler.method_6668("fabricServerEntityLoad");

			for (Load callback : callbacks) {
				profiler.method_6668(EventFactory.getHandlerName(callback));
				callback.onLoad(entity, world);
				profiler.method_6669();
			}

			profiler.method_6669();
		} else {
			for (Load callback : callbacks) {
				callback.onLoad(entity, world);
			}
		}
	});

	/**
	 * Called when Lightning strikes in a world.
	 */
	public static final Event<LightningStrike> LIGHTNING_STRIKE = EventFactory.createArrayBacked(LightningStrike.class, (listeners) -> (entity, world, x, y, z) -> {
		for (LightningStrike callback : listeners) {
			callback.onLightningStrike(entity, world, x, y, z);
		}
	});

	/**
	 * Called when an entity is hurt.
	 */
	public static final Event<EntityHurt> HURT = EventFactory.createArrayBacked(EntityHurt.class, (listeners) -> (entity, source, original, damage, attacker) -> {
		for (EntityHurt callback : listeners) {
			callback.entityHurt(entity, source, original, damage, attacker);
		}
	});

	/**
	 * Called when an entity is killed.
	 */
	public static final Event<EntityKilled> KILLED = EventFactory.createArrayBacked(EntityKilled.class, (listeners) -> (entity) -> {
		for (EntityKilled callback : listeners) {
			callback.entityKilled(entity);
		}
	});

	@FunctionalInterface
	public interface Load {
		void onLoad(class_1745 entity, class_1635 world);
	}

	@FunctionalInterface
	public interface LightningStrike {
		void onLightningStrike(class_1890 entity, class_99 world, double x, double y, double z);
	}

	@FunctionalInterface
	public interface EntityHurt {
		void entityHurt(class_1752 entity, class_1733 damageSource, float originalHealth, float damage, @Nullable class_1745 attacker);
	}

	@FunctionalInterface
	public interface EntityKilled {
		void entityKilled(class_1745 killed);
	}
}
