/*
 * Copyright (c) 2020 - 2021 Legacy Fabric
 * Copyright (c) 2016 - 2021 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.network;

import java.lang.ref.WeakReference;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.function.Consumer;

import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import com.google.common.collect.Sets;
import net.fabricmc.fabric.api.event.network.C2SPacketTypeCallback;
import net.fabricmc.fabric.api.network.PacketContext;
import net.fabricmc.fabric.api.network.PacketIdentifier;
import net.fabricmc.fabric.api.network.ServerSidePacketRegistry;
import net.minecraft.class_1435;
import net.minecraft.class_1457;
import net.minecraft.class_1488;
import net.minecraft.class_1591;
import net.minecraft.class_1637;
import net.minecraft.class_1643;
import net.minecraft.class_1963;

public class ServerSidePacketRegistryImpl extends PacketRegistryImpl implements ServerSidePacketRegistry {
	private final WeakHashMap<class_1963, Collection<String>> playerPayloadIds = new WeakHashMap<>();
	private final Set<WeakReference<class_1643>> handlers = Sets.newHashSet();

	public void onQueryResponse(class_1591 packet) {
	}

	public void addNetworkHandler(class_1643 handler) {
		this.handlers.add(new WeakReference<>(handler));
	}

	protected void forEachHandler(Consumer<class_1643> consumer) {
		Iterator<WeakReference<class_1643>> it = this.handlers.iterator();

		while (it.hasNext()) {
			class_1643 server = it.next().get();

			if (server != null) {
				consumer.accept(server);
			} else {
				it.remove();
			}
		}
	}

	@Override
	public boolean canPlayerReceive(class_1963 player, String id) {
		Collection<String> ids = this.playerPayloadIds.get(player);

		if (ids != null) {
			return ids.contains(id);
		} else {
			return false;
		}
	}

	@Override
	public boolean canPlayerReceive(class_1963 player, PacketIdentifier id) {
		return this.canPlayerReceive(player, id.toString());
	}

	@Override
	public void sendToPlayer(class_1963 player, class_1457<?> packet, GenericFutureListener<? extends Future<? super Void>> completionListener) {
		if (!(player instanceof class_1637)) {
			throw new RuntimeException("Can only send to ServerPlayerEntities!");
		} else {
			if (completionListener != null) {
				((class_1637) player).field_6639.field_6718.method_5099(packet, completionListener);
				return;
			}

			((class_1637) player).field_6639.method_6127(packet);
		}
	}

	@Override
	protected void onRegister(String id) {
		this.createRegisterTypePacket(PacketTypes.REGISTER, Collections.singleton(id)).ifPresent((packet) -> this.forEachHandler((n) -> n.method_6127(packet)));
	}

	@Override
	protected void onUnregister(String id) {
		this.createRegisterTypePacket(PacketTypes.UNREGISTER, Collections.singleton(id)).ifPresent((packet) -> this.forEachHandler((n) -> n.method_6127(packet)));
	}

	@Override
	protected Collection<String> getIdCollectionFor(PacketContext context) {
		return this.playerPayloadIds.computeIfAbsent(context.getPlayer(), (p) -> new HashSet<>());
	}

	@Override
	protected void onReceivedRegisterPacket(PacketContext context, Collection<String> ids) {
		C2SPacketTypeCallback.REGISTERED.invoker().accept(context.getPlayer(), ids);
	}

	@Override
	protected void onReceivedUnregisterPacket(PacketContext context, Collection<String> ids) {
		C2SPacketTypeCallback.UNREGISTERED.invoker().accept(context.getPlayer(), ids);
	}

	@Override
	public class_1457<?> toPacket(String id, class_1435 buf) {
		return new class_1488(id, buf);
	}

	@Override
	public class_1457<?> toPacket(PacketIdentifier id, class_1435 buf) {
		return this.toPacket(id.toString(), buf);
	}
}
