/*
 * Copyright (c) 2020 - 2021 Legacy Fabric
 * Copyright (c) 2016 - 2021 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.legacyfabric.fabric.api.networking.v1;

import java.util.Objects;
import net.minecraft.class_1435;
import net.minecraft.class_1457;
import net.minecraft.class_1605;
import io.netty.channel.ChannelFutureListener;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;

/**
 * Represents something that supports sending packets to channels.
 *
 * @see PacketByteBufs
 */
public interface PacketSender {
	/**
	 * Makes a packet for a channel.
	 *
	 * @param channelName the id of the channel
	 * @param buf         the content of the packet
	 */
	class_1457<?> createPacket(String channelName, class_1435 buf);

	/**
	 * Makes a packet for a channel.
	 *
	 * @param channelId the id of the channel
	 * @param buf       the content of the packet
	 */
	default class_1457<?> createPacket(class_1605 channelId, class_1435 buf) {
		return this.createPacket(channelId.toString(), buf);
	}

	/**
	 * Sends a packet.
	 *
	 * @param packet the packet
	 */
	void sendPacket(class_1457<?> packet);

	/**
	 * Sends a packet.
	 *
	 * @param packet   the packet
	 * @param callback an optional callback to execute after the packet is sent, may be {@code null}. The callback may also accept a {@link ChannelFutureListener}.
	 */
	void sendPacket(class_1457<?> packet, GenericFutureListener<? extends Future<? super Void>> callback);

	/**
	 * Sends a packet to a channel.
	 *
	 * @param channel the id of the channel
	 * @param buf     the content of the packet
	 */
	default void sendPacket(String channel, class_1435 buf) {
		Objects.requireNonNull(channel, "Channel cannot be null");
		Objects.requireNonNull(buf, "Payload cannot be null");

		this.sendPacket(this.createPacket(channel, buf));
	}

	/**
	 * Sends a packet to a channel.
	 *
	 * @param channelId the id of the channel
	 * @param buf       the content of the packet
	 */
	default void sendPacket(class_1605 channelId, class_1435 buf) {
		this.sendPacket(channelId.toString(), buf);
	}

	/**
	 * Sends a packet to a channel.
	 *
	 * @param channel  the id of the channel
	 * @param buf      the content of the packet
	 * @param callback an optional callback to execute after the packet is sent, may be {@code null}
	 */
	// the generic future listener can accept ChannelFutureListener
	default void sendPacket(String channel, class_1435 buf, GenericFutureListener<? extends Future<? super Void>> callback) {
		Objects.requireNonNull(channel, "Channel cannot be null");
		Objects.requireNonNull(buf, "Payload cannot be null");

		this.sendPacket(this.createPacket(channel, buf), callback);
	}

	/**
	 * Sends a packet to a channel.
	 *
	 * @param channel  the id of the channel
	 * @param buf      the content of the packet
	 * @param callback an optional callback to execute after the packet is sent, may be {@code null}
	 */
	// the generic future listener can accept ChannelFutureListener
	default void sendPacket(class_1605 channel, class_1435 buf, GenericFutureListener<? extends Future<? super Void>> callback) {
		this.sendPacket(channel.toString(), buf, callback);
	}
}
