/*
 * Copyright (c) 2020 - 2021 Legacy Fabric
 * Copyright (c) 2016 - 2021 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.legacyfabric.fabric.api.networking.v1;

import java.util.Objects;
import java.util.Set;
import net.minecraft.class_1435;
import net.minecraft.class_1457;
import net.minecraft.class_1605;
import net.minecraft.class_1637;
import net.minecraft.class_1643;
import net.minecraft.server.MinecraftServer;
import net.legacyfabric.fabric.impl.networking.server.ServerNetworkingImpl;
import net.legacyfabric.fabric.impl.networking.server.ServerPlayNetworkHandlerExtensions;

/**
 * Offers access to play stage server-side networking functionalities.
 *
 * <p>Server-side networking functionalities include receiving serverbound packets, sending clientbound packets, and events related to server-side network handlers.
 *
 * <p>This class should be only used for the logical server.
 */
public final class ServerPlayNetworking {
	/**
	 * Registers a handler to a channel.
	 * A global receiver is registered to all connections, in the present and future.
	 *
	 * <p>If a handler is already registered to the {@code channel}, this method will return {@code false}, and no change will be made.
	 * Use {@link #unregisterReceiver(class_1643, String)} to unregister the existing handler.
	 *
	 * @param channelName    the id of the channel
	 * @param channelHandler the handler
	 * @return false if a handler is already registered to the channel
	 * @see ServerPlayNetworking#unregisterGlobalReceiver(String)
	 * @see ServerPlayNetworking#registerReceiver(class_1643, String, PlayChannelHandler)
	 */
	public static boolean registerGlobalReceiver(String channelName, PlayChannelHandler channelHandler) {
		return ServerNetworkingImpl.PLAY.registerGlobalReceiver(channelName, channelHandler);
	}

	/**
	 * Registers a handler to a channel.
	 * A global receiver is registered to all connections, in the present and future.
	 *
	 * <p>If a handler is already registered to the {@code channel}, this method will return {@code false}, and no change will be made.
	 * Use {@link #unregisterReceiver(class_1643, class_1605)} to unregister the existing handler.
	 *
	 * @param channelId      the id of the channel
	 * @param channelHandler the handler
	 * @return false if a handler is already registered to the channel
	 * @see ServerPlayNetworking#unregisterGlobalReceiver(class_1605)
	 * @see ServerPlayNetworking#registerReceiver(class_1643, class_1605, PlayChannelHandler)
	 */
	public static boolean registerGlobalReceiver(class_1605 channelId, PlayChannelHandler channelHandler) {
		return registerGlobalReceiver(channelId.toString(), channelHandler);
	}

	/**
	 * Removes the handler of a channel.
	 * A global receiver is registered to all connections, in the present and future.
	 *
	 * <p>The {@code channel} is guaranteed not to have a handler after this call.
	 *
	 * @param channelName the id of the channel
	 * @return the previous handler, or {@code null} if no handler was bound to the channel
	 * @see ServerPlayNetworking#registerGlobalReceiver(String, PlayChannelHandler)
	 * @see ServerPlayNetworking#unregisterReceiver(class_1643, String)
	 */
	public static PlayChannelHandler unregisterGlobalReceiver(String channelName) {
		return ServerNetworkingImpl.PLAY.unregisterGlobalReceiver(channelName);
	}

	/**
	 * Removes the handler of a channel.
	 * A global receiver is registered to all connections, in the present and future.
	 *
	 * <p>The {@code channel} is guaranteed not to have a handler after this call.
	 *
	 * @param channelId the id of the channel
	 * @return the previous handler, or {@code null} if no handler was bound to the channel
	 * @see ServerPlayNetworking#registerGlobalReceiver(class_1605, PlayChannelHandler)
	 * @see ServerPlayNetworking#unregisterReceiver(class_1643, String)
	 */
	public static PlayChannelHandler unregisterGlobalReceiver(class_1605 channelId) {
		return unregisterGlobalReceiver(channelId.toString());
	}

	/**
	 * Gets all channel names which global receivers are registered for.
	 * A global receiver is registered to all connections, in the present and future.
	 *
	 * @return all channel names which global receivers are registered for.
	 */
	public static Set<String> getGlobalReceivers() {
		return ServerNetworkingImpl.PLAY.getChannels();
	}

	/**
	 * Registers a handler to a channel.
	 * This method differs from {@link ServerPlayNetworking#registerGlobalReceiver(String, PlayChannelHandler)} since
	 * the channel handler will only be applied to the player represented by the {@link class_1643}.
	 *
	 * <p>If a handler is already registered to the {@code channelName}, this method will return {@code false}, and no change will be made.
	 * Use {@link #unregisterReceiver(class_1643, String)} to unregister the existing handler.
	 *
	 * @param networkHandler the handler
	 * @param channelName    the id of the channel
	 * @param channelHandler the handler
	 * @return false if a handler is already registered to the channel name
	 * @see ServerPlayConnectionEvents#INIT
	 */
	public static boolean registerReceiver(class_1643 networkHandler, String channelName, PlayChannelHandler channelHandler) {
		Objects.requireNonNull(networkHandler, "Network handler cannot be null");

		return ((ServerPlayNetworkHandlerExtensions) networkHandler).getAddon().registerChannel(channelName, channelHandler);
	}

	/**
	 * Registers a handler to a channel.
	 * This method differs from {@link ServerPlayNetworking#registerGlobalReceiver(class_1605, PlayChannelHandler)} since
	 * the channel handler will only be applied to the player represented by the {@link class_1643}.
	 *
	 * <p>If a handler is already registered to the {@code channelId}, this method will return {@code false}, and no change will be made.
	 * Use {@link #unregisterReceiver(class_1643, class_1605)} to unregister the existing handler.
	 *
	 * @param networkHandler the handler
	 * @param channelId      the id of the channel
	 * @param channelHandler the handler
	 * @return false if a handler is already registered to the channel name
	 * @see ServerPlayConnectionEvents#INIT
	 */
	public static boolean registerReceiver(class_1643 networkHandler, class_1605 channelId, PlayChannelHandler channelHandler) {
		return registerReceiver(networkHandler, channelId.toString(), channelHandler);
	}

	/**
	 * Removes the handler of a channel.
	 *
	 * <p>The {@code channelName} is guaranteed not to have a handler after this call.
	 *
	 * @param channelName the id of the channel
	 * @return the previous handler, or {@code null} if no handler was bound to the channel name
	 */
	public static PlayChannelHandler unregisterReceiver(class_1643 networkHandler, String channelName) {
		Objects.requireNonNull(networkHandler, "Network handler cannot be null");

		return ServerNetworkingImpl.getAddon(networkHandler).unregisterChannel(channelName);
	}

	/**
	 * Removes the handler of a channel.
	 *
	 * <p>The {@code channelId} is guaranteed not to have a handler after this call.
	 *
	 * @param channelId the id of the channel
	 * @return the previous handler, or {@code null} if no handler was bound to the channel name
	 */
	public static PlayChannelHandler unregisterReceiver(class_1643 networkHandler, class_1605 channelId) {
		return unregisterReceiver(networkHandler, channelId.toString());
	}

	/**
	 * Gets all the channel names that the server can receive packets on.
	 *
	 * @param player the player
	 * @return All the channel names that the server can receive packets on
	 */
	public static Set<String> getReceived(class_1637 player) {
		Objects.requireNonNull(player, "Server player entity cannot be null");

		return getReceived(player.field_6639);
	}

	/**
	 * Gets all the channel names that the server can receive packets on.
	 *
	 * @param handler the network handler
	 * @return All the channel names that the server can receive packets on
	 */
	public static Set<String> getReceived(class_1643 handler) {
		Objects.requireNonNull(handler, "Server play network handler cannot be null");

		return ServerNetworkingImpl.getAddon(handler).getReceivableChannels();
	}

	/**
	 * Gets all channel names that the connected client declared the ability to receive a packets on.
	 *
	 * @param player the player
	 * @return All the channel names the connected client declared the ability to receive a packets on
	 */
	public static Set<String> getSendable(class_1637 player) {
		Objects.requireNonNull(player, "Server player entity cannot be null");

		return getSendable(player.field_6639);
	}

	/**
	 * Gets all channel names that a the connected client declared the ability to receive a packets on.
	 *
	 * @param handler the network handler
	 * @return True if the connected client has declared the ability to receive a packet on the specified channel
	 */
	public static Set<String> getSendable(class_1643 handler) {
		Objects.requireNonNull(handler, "Server play network handler cannot be null");

		return ServerNetworkingImpl.getAddon(handler).getSendableChannels();
	}

	/**
	 * Checks if the connected client declared the ability to receive a packet on a specified channel name.
	 *
	 * @param player      the player
	 * @param channelName the channel name
	 * @return True if the connected client has declared the ability to receive a packet on the specified channel
	 */
	public static boolean canSend(class_1637 player, String channelName) {
		Objects.requireNonNull(player, "Server player entity cannot be null");

		return canSend(player.field_6639, channelName);
	}

	/**
	 * Checks if the connected client declared the ability to receive a packet on a specified channel name.
	 *
	 * @param handler     the network handler
	 * @param channelName the channel name
	 * @return True if the connected client has declared the ability to receive a packet on the specified channel
	 */
	public static boolean canSend(class_1643 handler, String channelName) {
		Objects.requireNonNull(handler, "Server play network handler cannot be null");
		Objects.requireNonNull(channelName, "Channel name cannot be null");

		return ServerNetworkingImpl.getAddon(handler).getSendableChannels().contains(channelName);
	}

	/**
	 * Creates a packet which may be sent to a the connected client.
	 *
	 * @param channelName the channel name
	 * @param buf         the packet byte buf which represents the payload of the packet
	 * @return a new packet
	 */
	public static class_1457<?> createS2CPacket(String channelName, class_1435 buf) {
		Objects.requireNonNull(channelName, "Channel cannot be null");
		Objects.requireNonNull(buf, "Buf cannot be null");

		return ServerNetworkingImpl.createPlayC2SPacket(channelName, buf);
	}

	/**
	 * Creates a packet which may be sent to a the connected client.
	 *
	 * @param channelId the channel name
	 * @param buf       the packet byte buf which represents the payload of the packet
	 * @return a new packet
	 */
	public static class_1457<?> createS2CPacket(class_1605 channelId, class_1435 buf) {
		return createS2CPacket(channelId.toString(), buf);
	}

	/**
	 * Gets the packet sender which sends packets to the connected client.
	 *
	 * @param player the player
	 * @return the packet sender
	 */
	public static PacketSender getSender(class_1637 player) {
		Objects.requireNonNull(player, "Server player entity cannot be null");

		return getSender(player.field_6639);
	}

	/**
	 * Gets the packet sender which sends packets to the connected client.
	 *
	 * @param handler the network handler, representing the connection to the player/client
	 * @return the packet sender
	 */
	public static PacketSender getSender(class_1643 handler) {
		Objects.requireNonNull(handler, "Server play network handler cannot be null");

		return ServerNetworkingImpl.getAddon(handler);
	}

	/**
	 * Sends a packet to a player.
	 *
	 * @param player      the player to send the packet to
	 * @param channelName the channel of the packet
	 * @param buf         the payload of the packet.
	 */
	public static void send(class_1637 player, String channelName, class_1435 buf) {
		Objects.requireNonNull(player, "Server player entity cannot be null");
		Objects.requireNonNull(channelName, "Channel name cannot be null");
		Objects.requireNonNull(buf, "Packet byte buf cannot be null");

		player.field_6639.method_6127(createS2CPacket(channelName, buf));
	}

	/**
	 * Sends a packet to a player.
	 *
	 * @param player    the player to send the packet to
	 * @param channelId the channel of the packet
	 * @param buf       the payload of the packet.
	 */
	public static void send(class_1637 player, class_1605 channelId, class_1435 buf) {
		send(player, channelId.toString(), buf);
	}

	// Helper methods

	/**
	 * Returns the <i>Minecraft</i> Server of a server play network handler.
	 *
	 * @param handler the server play network handler
	 */
	public static MinecraftServer getServer(class_1643 handler) {
		Objects.requireNonNull(handler, "Network handler cannot be null");

		return handler.field_6719.field_6645;
	}

	private ServerPlayNetworking() {
	}

	@FunctionalInterface
	public interface PlayChannelHandler {
		/**
		 * Handles an incoming packet.
		 *
		 * <p>This method is executed on {@linkplain io.netty.channel.EventLoop netty's event loops}.
		 * Modification to the game should be {@linkplain net.minecraft.class_1716#method_6733(Runnable)}  scheduled} using the provided Minecraft server instance.
		 *
		 * <p>An example usage of this is to create an explosion where the player is looking:
		 * <pre>{@code
		 * ServerPlayNetworking.registerReceiver(new String("mymod", "boom"), (server, player, handler, buf, responseSender) -&rt; {
		 * 	boolean fire = buf.readBoolean();
		 *
		 * 	// All operations on the server or world must be executed on the server thread
		 * 	server.execute(() -&rt; {
		 * 		ModPacketHandler.createExplosion(player, fire);
		 *    });
		 * });
		 * }</pre>
		 *
		 * @param server         the server
		 * @param player         the player
		 * @param handler        the network handler that received this packet, representing the player/client who sent the packet
		 * @param buf            the payload of the packet
		 * @param responseSender the packet sender
		 */
		void receive(MinecraftServer server, class_1637 player, class_1643 handler, class_1435 buf, PacketSender responseSender);
	}
}
